package javarequirementstracer.mojo;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import javarequirementstracer.JavaRequirementsTracerBean;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;

/**
 * @author Ruud de Jong
 * @goal trace
 * @phase test
 * @requiresDependencyResolution test
 */
public class JrtMojo extends AbstractMojo {

	private static final String REPORTER_PROPERTIES_FILENAME = "/reporter.properties";
	private static final String REPORTER_GROUPID = "reporter.groupid";
	private static final String REPORTER_ARTIFACTID = "reporter.artifactid";
	private static final String REPORTER_VERSION = "reporter.version";
	private static final String REPORTER_SCOPE = "runtime";
	private static final String REPORTER_TYPE = "jar";


	/**
	 * Properties file containing all tuning parameters for a run.
	 * Default: {@value JavaRequirementsTracerBean#DEFAULT_PARAMS_FILENAME}.
	 * If the file does not exist the plugin will skip generating a report and print a log message. 
	 * 
	 * @parameter expression="${paramsFilename}"
	 */
	private File paramsFilename;

	/**
	 * Properties file containing all expected requirements labels.
	 * Default: {@value JavaRequirementsTracerBean#DEFAULT_EXPECTED_LABELS_FILENAME}.
	 * 
	 * @parameter expression="${labelsFilename}"
	 */
	private File labelsFilename;
	
	/**
	 * Report filename. Default: {@value JavaRequirementsTracerBean#DEFAULT_REPORT_FILENAME}.
	 * 
	 * @parameter expression="${reportFilename}"
	 */
	private File reportFilename;

	/**
	 * The build number to display in the report.
	 * 
	 * @parameter expression="${buildNumber}"
	 */
	private String buildNumber;

	/**
	 * The Maven Project Object
	 *
	 * @parameter expression="${project}"
	 * @required
	 * @readonly
	 */
	private MavenProject project;

	/**
	 * The classpath elements of the project being tested.
	 *
	 * @parameter expression="${project.testClasspathElements}"
	 * @required
	 * @readonly
	 */
	private List<String> testClasspathElements;

	/**
	 * ArtifactRepository of the localRepository. To obtain the directory of localRepository in unit tests use
	 * System.setProperty( "localRepository").
	 *
	 * @parameter expression="${localRepository}"
	 * @required
	 * @readonly
	 */
	private ArtifactRepository localRepository;

	/**
	 * ArtifactFactory component.
	 *
	 * @component
	 */
	private ArtifactFactory artifactFactory;

	/**
	 * Maven plugin execute method.
	 * This will check whether the reporter should be run,
	 * assembles the classloader and start the reporter from the alternate classloader.
	 * @throws MojoExecutionException when an error occurs.
	 */
	public final void execute() throws MojoExecutionException {
		getLog().info("Maven JRT Plugin");
		getLog().info("Params filename: " + this.paramsFilename);
		getLog().info("Labels filename: " + this.labelsFilename);
		getLog().info("Report filename: " + this.reportFilename);
		getLog().info("Buildnumber: " + this.buildNumber);
		getLog().info("Project basedir: " + this.project.getBasedir());

		if (parametersValid()) {
			List<URL> classpathUrlList = new ArrayList<URL>();
			classpathUrlList.add(getReporterArtifactUrl());
			classpathUrlList.addAll(getTestClasspathUrls());

			URLClassLoader classLoader = new URLClassLoader(classpathUrlList.toArray(new URL[classpathUrlList.size()]));

			runTracerWithClassloader(classLoader);
		}
	}

	/**
	 * Checks the input parameters.
	 * If paramFilename is null, the default will be used.
	 * But paramFilename MUST exist! Otherwise, this plugin will do nothing.
	 * @return true is all input parameters are okay / plugin has work to do.
	 */
	private boolean parametersValid() {
		File paramFile = this.paramsFilename;
		if (paramFile == null) {
			paramFile = new File(this.project.getBasedir() + "/" + JavaRequirementsTracerBean.DEFAULT_PARAMS_FILENAME);
		}
		if (!paramFile.exists()) {
			getLog().info("Skipping traceability. No params file '" + paramFile.getAbsolutePath() + "' found.");
			return false;
		}
		return true;
	}

	/**
	 * Runs the Java Requirements Tracer.
	 * This tracer needs to be loaded by the alternative classloader, which
	 * has all the right classpaths and dependencies added.
	 * This can ONLY be done through reflection, because the same class from different
	 * classloaders will ALWAYS be not-equal AND will ALWAYS give class cast exceptions.
	 * @param classLoader the Class Loader to use.
	 * @throws MojoExecutionException when shit hits the fan.
	 */
	private void runTracerWithClassloader(ClassLoader classLoader) throws MojoExecutionException {
		try {
			Class tracerClass = classLoader.loadClass(JavaRequirementsTracerBean.class.getName());
			Object tracerInstance = tracerClass.getConstructor().newInstance();
			Method setClassLoaderMethod = tracerClass.getDeclaredMethod("setClassLoader", ClassLoader.class);
			setClassLoaderMethod.invoke(tracerInstance, classLoader);
			invokeFileSetter(tracerInstance, "setBaseDir", this.project.getBasedir());
			invokeStringSetter(tracerInstance, "setParamsFilename", this.paramsFilename);
			invokeStringSetter(tracerInstance, "setExpectedLabelsFilename", this.labelsFilename);
			invokeStringSetter(tracerInstance, "setReportFileName", this.reportFilename);
			invokeStringSetter(tracerInstance, "setBuildNumber", this.buildNumber);
			Method runMethod = tracerClass.getDeclaredMethod("run");
			runMethod.invoke(tracerInstance);
		} catch (ClassNotFoundException e) {
			throw new MojoExecutionException("Could not load " + JavaRequirementsTracerBean.class.getName() + " from the alternate classloader.", e);
		} catch (NoSuchMethodException e) {
			throw new MojoExecutionException("Could not find constructor of " + JavaRequirementsTracerBean.class.getName() + ".", e);
		} catch (InvocationTargetException e) {
			throw new MojoExecutionException("Could not find invoke constructor of " + JavaRequirementsTracerBean.class.getName() + ".", e);
		} catch (IllegalAccessException e) {
			throw new MojoExecutionException("Could not access constructor of " + JavaRequirementsTracerBean.class.getName() + ".", e);
		} catch (InstantiationException e) {
			throw new MojoExecutionException("Could not create instance of " + JavaRequirementsTracerBean.class.getName() + ".", e);
		}
	}

	private void invokeFileSetter(Object instance, String methodName, File fileValue) throws NoSuchMethodException, InvocationTargetException, IllegalAccessException {
		Method fileSetterMethod = instance.getClass().getDeclaredMethod(methodName, File.class);
		fileSetterMethod.invoke(instance, fileValue);
	}

	private void invokeStringSetter(Object instance, String methodName, File fileValue) throws NoSuchMethodException, InvocationTargetException, IllegalAccessException {
		if (fileValue != null) {
			invokeStringSetter(instance, methodName, fileValue.getAbsolutePath());
		}
	}

	private void invokeStringSetter(Object instance, String methodName, String value) throws NoSuchMethodException, InvocationTargetException, IllegalAccessException {
		if (value != null) {
			Method stringSetterMethod = instance.getClass().getDeclaredMethod(methodName, String.class);
			stringSetterMethod.invoke(instance, value);
		}
	}

	/**
	 * Convert the test class path elements into URLs.
	 * @return a list of URLs.
	 * @throws MojoExecutionException when URLs are invalid.
	 */
	private List<URL> getTestClasspathUrls() throws MojoExecutionException {
		List<URL> urlList = new ArrayList<URL>();
		for (String testClasspath : this.testClasspathElements) {
			URL url = pathToUrl(testClasspath);
			urlList.add(url);
		}
		return urlList;
	}

	/**
	 * Convert a String to a File to an URL.
	 * @param path the string-form of a file or path reference to convert.
	 * @return the URL.
	 * @throws MojoExecutionException then the URL is invalid.
	 */
	private URL pathToUrl(String path) throws MojoExecutionException{
		try {
			File file = new File(path);
			return file.toURI().toURL();
		} catch (MalformedURLException e) {
			throw new MojoExecutionException("Invalid URL: " + path, e);
		}
	}

	/**
	 * Retrieve the URL of the reporter artifact.
	 * This reads the properties file from the JAR of this MOJO and consult the artifact factory for the URL.
	 * @return the reporter URL.
	 * @throws MojoExecutionException when there are problems with the properties or the URL.
	 */
	private URL getReporterArtifactUrl() throws MojoExecutionException {
		InputStream propertiesStream = JrtMojo.class.getResourceAsStream(REPORTER_PROPERTIES_FILENAME);
		if (propertiesStream == null) {
			throw new MojoExecutionException("Cannot find properties file: " + REPORTER_PROPERTIES_FILENAME);
		}
		Properties reporterProperties = new Properties();
		try {
			reporterProperties.load(propertiesStream);
		} catch (IOException e) {
			throw new MojoExecutionException("Cannot load properties from file: " + REPORTER_PROPERTIES_FILENAME, e);
		}
		Artifact reporterArtifact = this.artifactFactory.createArtifact(
				reporterProperties.getProperty(REPORTER_GROUPID),
				reporterProperties.getProperty(REPORTER_ARTIFACTID),
				reporterProperties.getProperty(REPORTER_VERSION), REPORTER_SCOPE, REPORTER_TYPE);
		URL url = pathToUrl(this.localRepository.getBasedir() + "/" + this.localRepository.pathOf(reporterArtifact));
		getLog().info("Reporter URL dependency: " + url.toString());
		return url;
	}
}